<?php

namespace App\Http\Controllers;

use App\Models\MachineReading;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class MachineController extends Controller
{
    /**
     * Display the machine monitoring page.
     */
    public function index(): View
        {
            // Get the latest reading for each machine_id by recorded_at (not by ID)
            $latestReadings = $this->latestReadingsQuery()
                ->get();
        
            $machines = $latestReadings->map(function ($reading) {
                return [
                    'id' => $reading->id,
                    'name' => $reading->machine_id, // VARCHAR
                    'amp' => $reading->amp,
                    'hm' => $reading->hm,
                    'temp' => $reading->temp,
                    'moist' => $reading->moist,
                    'last_updated' => $reading->recorded_at
                        ? Carbon::parse($reading->recorded_at)->format('Y-m-d H:i:s')
                        : '-',
                    'status' => 'active',
                ];
            });
        
            $machines = $machines
                ->sortBy('name', SORT_NATURAL | SORT_FLAG_CASE)
                ->values();
        
            // If no data yet, show empty or some default
            if ($machines->isEmpty()) {
                $machines = collect([]);
            }
        
            return view('machines.index', compact('machines'));
        }

    /**
     * Get machine data for AJAX updates with filtering
     */
    public function getData(Request $request) {
        $hasCustomRange = $request->filled('start') && $request->filled('end');

        if ($hasCustomRange) {
            [$startDate, $endDate, $error] = $this->resolveRange(
                $request,
                ['1h' => 1, '8h' => 8, '24h' => 24],
                24
            );

            if ($error) {
                return response()->json([
                    'error' => 'Invalid date range',
                    'message' => $error,
                ], 422);
            }

            $latestReadings = $this->latestReadingsQuery($startDate, $endDate)->get();
        } else {
            $startDate = null;
            $endDate = null;
            $latestReadings = $this->latestReadingsQuery()->get();
        }

        $machines = $latestReadings->map(function($reading) {
            return [
                'id' => $reading->id,
                'name' => $reading->machine_id,
                'amp' => $reading->amp,
                'hm' => $reading->hm,
                'temp' => $reading->temp,
                'moist' => $reading->moist,
                'last_updated' => $reading->recorded_at ? Carbon::parse($reading->recorded_at)->format('Y-m-d H:i:s') : '-',
                'status' => 'active',
            ];
        });
        
        return response()->json([
            'data' => $machines,
            'range' => [
                'start' => $startDate ? $startDate->format('Y-m-d H:i:s') : null,
                'end' => $endDate ? $endDate->format('Y-m-d H:i:s') : null
            ]
        ]);
    }
    public function tableData(Request $req)
    {
        [$startDate, $endDate, $error] = $this->resolveRange(
            $req,
            ['1h' => 1, '2h' => 2, '4h' => 4, '8h' => 8, '16h' => 16, '24h' => 24],
            1
        );

        if ($error) {
            return response()->json([
                'error' => 'Invalid date range',
                'message' => $error,
            ], 422);
        }

        $rows = MachineReading::query()
            ->selectRaw('
                machine_id, 
                AVG(amp) as amp, 
                MAX(hm) as hm, 
                AVG(temp) as temp, 
                AVG(moist) as moist, 
                MAX(recorded_at) as last_updated
            ')
            ->whereBetween('recorded_at', [$startDate, $endDate])
            ->groupBy('machine_id')
            ->get()
            ->map(function($r) {
                return [
                    'name' => $r->machine_id,
                    'last_updated' => $r->last_updated ? Carbon::parse($r->last_updated)->format('Y-m-d H:i:s') : '-',
                    'amp' => round($r->amp, 2),
                    'hm' => (float)$r->hm,
                    'temp' => round($r->temp, 1),
                    'moist' => round($r->moist, 1),
                    'status' => 'active'
                ];
            });

        // 3. Custom Sorting Logic
        $sortBy = $req->query('sort_by', 'auto');
        if ($sortBy === 'auto') {
            $rows = $rows->sort(function($a, $b) {
                // Priority 1: HM ASC
                if ($a['hm'] != $b['hm']) {
                    return $a['hm'] <=> $b['hm'];
                }

                // Priority 2: Ampere < 14 first, then value ASC
                $aUnder = $a['amp'] < 14;
                $bUnder = $b['amp'] < 14;
                if ($aUnder && !$bUnder) return -1;
                if (!$aUnder && $bUnder) return 1;
                if ($a['amp'] != $b['amp']) {
                    return $a['amp'] <=> $b['amp'];
                }

                // Priority 3: Temp deviation from 60 DESC, value ASC
                $aTempDev = abs($a['temp'] - 60);
                $bTempDev = abs($b['temp'] - 60);
                if (round($aTempDev, 1) != round($bTempDev, 1)) {
                    return $bTempDev <=> $aTempDev;
                }
                if ($a['temp'] != $b['temp']) {
                    return $a['temp'] <=> $b['temp'];
                }

                // Priority 4: Moist deviation from 20 DESC, value ASC
                $aMoistDev = abs($a['moist'] - 20);
                $bMoistDev = abs($b['moist'] - 20);
                if (round($aMoistDev, 1) != round($bMoistDev, 1)) {
                    return $bMoistDev <=> $aMoistDev;
                }
                if ($a['moist'] != $b['moist']) {
                    return $a['moist'] <=> $b['moist'];
                }

                return $a['name'] <=> $b['name'];
            });
        } elseif ($sortBy === 'amp') {
            $rows = $rows->sort(function($a, $b) {
                $aUnder = $a['amp'] < 14;
                $bUnder = $b['amp'] < 14;
                if ($aUnder && !$bUnder) return -1;
                if (!$aUnder && $bUnder) return 1;
                if ($a['amp'] != $b['amp']) {
                    return $a['amp'] <=> $b['amp'];
                }
                return $a['name'] <=> $b['name'];
            });
        } elseif ($sortBy === 'hm') {
            $rows = $rows->sort(function($a, $b) {
                if ($a['hm'] != $b['hm']) {
                    return $a['hm'] <=> $b['hm'];
                }
                return $a['name'] <=> $b['name'];
            });
        } elseif ($sortBy === 'temp') {
            $rows = $rows->sort(function($a, $b) {
                $distA = abs($a['temp'] - 60);
                $distB = abs($b['temp'] - 60);
                if (round($distA, 1) != round($distB, 1)) return $distB <=> $distA;
                if ($a['temp'] != $b['temp']) return $a['temp'] <=> $b['temp'];
                return $a['name'] <=> $b['name'];
            });
        } elseif ($sortBy === 'moist') {
            $rows = $rows->sort(function($a, $b) {
                $distA = abs($a['moist'] - 20);
                $distB = abs($b['moist'] - 20);
                if (round($distA, 1) != round($distB, 1)) return $distB <=> $distA;
                if ($a['moist'] != $b['moist']) return $a['moist'] <=> $b['moist'];
                return $a['name'] <=> $b['name'];
            });
        } else {
            $rows = $rows->sortBy('name');
        }

        return response()->json([
            'data' => $rows->values(),
            'summary_info' => [
                'start' => $startDate->format('Y-m-d H:i:s'),
                'end' => $endDate->format('Y-m-d H:i:s'),
                'is_summary' => true,
                'current_sort' => $sortBy
            ]
        ]);
    }

    private function latestReadingsQuery(?Carbon $startDate = null, ?Carbon $endDate = null)
    {
        $latestPerMachine = MachineReading::query()
            ->select('machine_id', DB::raw('MAX(recorded_at) as max_recorded_at'))
            ->when($startDate && $endDate, function($q) use ($startDate, $endDate) {
                $q->whereBetween('recorded_at', [$startDate, $endDate]);
            })
            ->groupBy('machine_id');

        return MachineReading::query()
            ->from('machine_readings as mr')
            ->select('mr.*')
            ->joinSub($latestPerMachine, 'latest', function($join) {
                $join->on('mr.machine_id', '=', 'latest.machine_id')
                    ->on('mr.recorded_at', '=', 'latest.max_recorded_at');
            })
            ->when($startDate && $endDate, function($q) use ($startDate, $endDate) {
                $q->whereBetween('mr.recorded_at', [$startDate, $endDate]);
            })
            ->whereIn('mr.id', function($q) {
                $q->select(DB::raw('MAX(id)'))
                    ->from('machine_readings as mr2')
                    ->whereColumn('mr2.machine_id', 'mr.machine_id')
                    ->whereColumn('mr2.recorded_at', 'mr.recorded_at');
            });
    }

    private function resolveRange(Request $request, array $filterHours, int $defaultHours): array
    {
        $start = $request->input('start');
        $end = $request->input('end');

        if (($start && !$end) || (!$start && $end)) {
            return [null, null, 'Start and end must be provided together.'];
        }

        if ($start && $end) {
            try {
                $startDate = $this->parseDateTime($start);
                $endDate = $this->parseDateTime($end);
            } catch (\Throwable $e) {
                return [null, null, 'Start/end format is invalid.'];
            }

            if ($startDate->greaterThanOrEqualTo($endDate)) {
                return [null, null, 'Start must be before end.'];
            }

            $now = now();
            if ($startDate->greaterThan($now)) {
                return [null, null, 'Start must not be in the future.'];
            }
            if ($endDate->greaterThan($now)) {
                return [null, null, 'End must not be in the future.'];
            }

            return [$startDate, $endDate, null];
        }

        $filter = $request->input('filter');
        if ($filter) {
            $shiftRange = $this->resolveShiftRange($filter, now());
            if ($shiftRange) {
                return [$shiftRange[0], $shiftRange[1], null];
            }
        }
        $hours = $filterHours[$filter] ?? $defaultHours;
        $endDate = now();
        $startDate = now()->subHours($hours);

        return [$startDate, $endDate, null];
    }

    private function resolveShiftRange(string $filter, Carbon $now): ?array
    {
        $shiftStartHours = [
            'shift1' => 7,
            'shift2' => 15,
            'shift3' => 23,
        ];

        if (!array_key_exists($filter, $shiftStartHours)) {
            return null;
        }

        $start = $now->copy()->setTime($shiftStartHours[$filter], 0, 0);
        if ($now->lt($start)) {
            $start->subDay();
        }

        $end = $start->copy()->addHours(8)->subMinute();
        if ($end->greaterThan($now)) {
            $end = $now->copy();
        }

        return [$start, $end];
    }

    private function parseDateTime(string $value): Carbon
    {
        try {
            $parsed = Carbon::createFromFormat('Y-m-d H:i', $value);
            if ($parsed !== false) {
                return $parsed;
            }
        } catch (\Throwable $e) {
            return Carbon::parse($value);
        }

        return Carbon::parse($value);
    }
}




