<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\MachineReading;
use Carbon\Carbon;
use Illuminate\Http\Request;

class MachineReadingController extends Controller
{
    public function store(Request $request)
    {
        $validated = $request->validate([
            'machine_id' => 'required|string',
            'amp'        => 'required|numeric',
            'hm'         => 'required|numeric',
            'temp'       => 'required|numeric',
            'moist'      => 'required|numeric',
            'timestamp'  => 'nullable',
        ]);

        $recordedAt = now();

        if ($request->filled('timestamp')) {
            try {
                $recordedAt = $this->parseTimestamp($request->input('timestamp'));
            } catch (\Throwable $e) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid timestamp format',
                ], 422);
            }
        }

        $reading = MachineReading::create([
            'machine_id'  => $validated['machine_id'],
            'amp'         => $validated['amp'],
            'hm'          => $validated['hm'],
            'temp'        => $validated['temp'],
            'moist'       => $validated['moist'],
            'recorded_at' => $recordedAt,
        ]);

        return response()->json([
            'status'  => 'success',
            'message' => 'Data recorded successfully',
            'data'    => $reading
        ], 201);
    }

    private function parseTimestamp(string|int|float $value): Carbon
    {
        if (is_numeric($value)) {
            $numeric = (float) $value;
            if ($numeric >= 1000000000000) {
                $numeric = $numeric / 1000;
            }
            return Carbon::createFromTimestamp($numeric);
        }

        return Carbon::parse((string) $value);
    }
}
